<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\WishlistItem;
use App\Models\UserLink;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class WishlistItemController extends Controller
{
    /**
     * Get all wishlist items for a specific link
     */
    public function index($linkId)
    {
        // First check if the link belongs to the authenticated user or is public
        $link = UserLink::findOrFail($linkId);
        
        // If not authenticated or link doesn't belong to user, only return if it's a public request
        if (!Auth::check() || $link->user_id !== Auth::id()) {
            // For public access, we don't need to check anything else
            // The link ID is already validated above
        } else {
            // For authenticated users, ensure they own the link
            $link = UserLink::where('user_id', Auth::id())
                ->where('id', $linkId)
                ->where('category', 'wishlist')
                ->firstOrFail();
        }
        
        $items = WishlistItem::where('user_link_id', $linkId)
            ->orderBy('created_at', 'desc')
            ->get();
            
        return response()->json($items);
    }
    
    /**
     * Store a new wishlist item
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'user_link_id' => 'required|exists:user_links,id',
            'item_name' => 'required|string|max:255',
            'item_url' => 'required|url|max:2048',
            'price' => 'nullable|numeric|min:0',
            'currency' => 'nullable|string|max:10',
        ]);
        
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }
        
        // Ensure the link belongs to the authenticated user and is a wishlist
        $link = UserLink::where('user_id', Auth::id())
            ->where('id', $request->user_link_id)
            ->where('category', 'wishlist')
            ->firstOrFail();
        
        $item = new WishlistItem();
        $item->user_link_id = $request->user_link_id;
        $item->item_name = $request->item_name;
        $item->item_url = $request->item_url;
        $item->price = $request->price;
        $item->currency = $request->currency;
        $item->save();
        
        return response()->json($item, 201);
    }
    
    /**
     * Update an existing wishlist item
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'user_link_id' => 'required|exists:user_links,id',
            'item_name' => 'required|string|max:255',
            'item_url' => 'required|url|max:2048',
            'price' => 'nullable|numeric|min:0',
            'currency' => 'nullable|string|max:10',
        ]);
        
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }
        
        // Ensure the link belongs to the authenticated user and is a wishlist
        $link = UserLink::where('user_id', Auth::id())
            ->where('id', $request->user_link_id)
            ->where('category', 'wishlist')
            ->firstOrFail();
        
        // Find the item and ensure it belongs to the specified link
        $item = WishlistItem::where('user_link_id', $request->user_link_id)
            ->findOrFail($id);
        
        $item->item_name = $request->item_name;
        $item->item_url = $request->item_url;
        $item->price = $request->price;
        $item->currency = $request->currency;
        $item->save();
        
        return response()->json($item);
    }
    
    /**
     * Delete a wishlist item
     */
    public function destroy($id)
    {
        // Find the item
        $item = WishlistItem::findOrFail($id);
        
        // Ensure the link belongs to the authenticated user
        $link = UserLink::where('user_id', Auth::id())
            ->where('id', $item->user_link_id)
            ->firstOrFail();
        
        $item->delete();
        
        return response()->json(null, 204);
    }
} 