<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\UserLink;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class UserLinkController extends Controller
{
    /**
     * Get all links for the authenticated user
     */
    public function index()
    {
        $links = UserLink::where('user_id', Auth::id())
            ->orderBy('link_order', 'asc')
            ->get();
            
        return response()->json($links);
    }
    
    /**
     * Get all links for a specific user by username
     */
    public function getUserLinks($username)
    {
        $user = \App\Models\User::where('username', $username)
            ->orWhere('name', 'LIKE', str_replace('-', '%', $username))
            ->firstOrFail();
            
        $links = UserLink::where('user_id', $user->id)
            ->orderBy('link_order', 'asc')
            ->get();
            
        return response()->json($links);
    }
    
    /**
     * Store a new link
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'url' => 'required|url|max:2048',
            'category' => 'required|in:social,donation,affiliate,wishlist',
            'affiliate_url' => 'nullable|url|max:2048',
        ]);
        
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }
        
        // Get the highest order value
        $maxOrder = UserLink::where('user_id', Auth::id())->max('link_order');
        
        $link = new UserLink();
        $link->user_id = Auth::id();
        $link->title = $request->title;
        $link->url = $request->url;
        $link->category = $request->category;
        $link->affiliate_url = $request->affiliate_url;
        $link->link_order = $maxOrder + 1;
        $link->save();
        
        return response()->json($link, 201);
    }
    
    /**
     * Update an existing link
     */
    public function update(Request $request, $id)
    {
        $link = UserLink::where('user_id', Auth::id())
            ->findOrFail($id);
            
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'url' => 'required|url|max:2048',
            'category' => 'required|in:social,donation,affiliate,wishlist',
            'affiliate_url' => 'nullable|url|max:2048',
        ]);
        
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }
        
        $link->title = $request->title;
        $link->url = $request->url;
        $link->category = $request->category;
        $link->affiliate_url = $request->affiliate_url;
        $link->save();
        
        return response()->json($link);
    }
    
    /**
     * Delete a link
     */
    public function destroy($id)
    {
        $link = UserLink::where('user_id', Auth::id())
            ->findOrFail($id);
            
        $link->delete();
        
        return response()->json(null, 204);
    }
    
    /**
     * Update the order of links
     */
    public function reorder(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'links' => 'required|array',
            'links.*.id' => 'required|exists:user_links,id',
            'links.*.link_order' => 'required|integer|min:0',
        ]);
        
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }
        
        foreach ($request->links as $linkData) {
            $link = UserLink::where('user_id', Auth::id())
                ->findOrFail($linkData['id']);
                
            $link->link_order = $linkData['link_order'];
            $link->save();
        }
        
        return response()->json(['message' => 'Links reordered successfully']);
    }
} 